import yaml
from sqlalchemy import create_engine, Engine
from sqlalchemy.orm import declarative_base, sessionmaker, Session
from typing import Type

class DatabaseManager:
    def __init__(self, config_path: str = "config.yaml") -> None:
        """
        データベースマネージャーを初期化し、データベースエンジンとセッションを設定する。

        Args:
            config_path (str): 設定ファイルのパス（デフォルトは "config.yaml"）。
        """
        # 設定ファイルからデータベースURLを取得
        with open(config_path, "r", encoding="utf-8") as file:
            config: dict = yaml.safe_load(file)

        db_url: str = config["database"]["url"]

        # データベースエンジンの作成
        self.engine: Engine = create_engine(db_url)

        # セッションの作成
        self.session_factory: sessionmaker[Session] = sessionmaker(
            autocommit=False, autoflush=False, bind=self.engine
        )

        # ベースモデルの作成
        self.Base: Type = declarative_base()

    def init_db(self) -> None:
        """
        データベースの初期化（テーブル作成など）。
        """
        self.Base.metadata.create_all(bind=self.engine)

    def get_session(self) -> Session:
        """
        新しいセッションを取得する。

        Returns:
            Session: SQLAlchemyのデータベースセッション。
        """
        return self.session_factory()
