from typing import List, Optional
import pandas as pd


def RCI(close: pd.Series, timeperiod: int = 9) -> List[Optional[float]]:
    """
    RCIを計算する関数。

    Args:
        close (pd.Series): 終値のSeries。
        timeperiod (int, optional): 計算期間（デフォルトは9）。

    Returns:
        List[Optional[float]]: 計算されたRCI値のリスト。期間に満たない場合はNone。
    """
    # RCIの値を格納するリスト
    rci = []

    # DataFrameを作成し、列名を"close"に設定
    # df = pd.DataFrame(close, columns=["Close"])

    for i in range(len(close)):
        if i < timeperiod:
            rci.append(None)  # 期間に満たない場合の値はNone
        else:
            d = 0
            # 現在の期間の価格に対応するランクを計算
            # (i - timeperiod + 1)行から(i+1)行目までの値を取得
            sliced_df = close.iloc[i - timeperiod + 1: i + 1]

            # Rankを計算（降順でランク付け）
            rank_price = sliced_df.rank(method="min", ascending=False).values

            # 日付のランクと終値のランクの差を2乗して合計
            for j in range(timeperiod):
                d += ((timeperiod - j) - rank_price[j]) ** 2

            # RCI計算式に適用
            rci_value = (
                1 - (6 * d / (timeperiod * (timeperiod**2 - 1)))) * 100
            rci.append(rci_value)

    return rci
